// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit

protocol SelectableItem {
    func textToDisplay() -> String
}

class SelectButton: UIButton {
    weak var presenter: UIViewController?
    var selectionTitle: String = ""
    var selectionMessage: String = ""
    var actionOnSelection: ((_ item: SelectableItem) -> Void)?
    
    private var list: [SelectableItem] = []

    override func awakeFromNib() {
        super.awakeFromNib()
        
        setupView()
        addTarget(self, action: #selector(onSelectButtonClick), for: .touchUpInside)
    }
    
    // MARK: Setter
    func setListForSelection(_ items: [SelectableItem], _ index: Int) {
        list = items
        let value = list[index]
        setTitle(value.textToDisplay(), for: .normal)
    }
    
    // MARK: IBAction
    @objc private func onSelectButtonClick() {
        let actionSheet = UIAlertController(title: selectionTitle, message: selectionMessage, preferredStyle: .actionSheet)
        let cancelAction = UIAlertAction(title: "btn_cancel".localized, style: .cancel) { [weak self] _ in
            self?.presenter?.dismiss(animated: true, completion: nil)
        }
        
        actionSheet.addAction(cancelAction)
        
        for item in list {
            let action = UIAlertAction(title: item.textToDisplay(), style: .default) { [weak self] _ in
                self?.setTitle(item.textToDisplay(), for: .normal)
                self?.actionOnSelection?(item)
                self?.presenter?.dismiss(animated: true, completion: nil)
            }
            
            actionSheet.addAction(action)
        }
        
        actionSheet.modalPresentationStyle = .popover
        if let popPresenter = actionSheet.popoverPresentationController {
            popPresenter.sourceView = self
            popPresenter.sourceRect = bounds
        }
        
        presenter?.present(actionSheet, animated: true, completion: nil)
    }
    
    func setupBorderColor() {
        layer.borderColor = UIColor.pinText?.cgColor
    }
    
    func setupView() {
        setupBorderColor()
        layer.borderWidth = 1.0
        
        transform = CGAffineTransform(scaleX: -1.0, y: 1.0)
        titleLabel?.transform = CGAffineTransform(scaleX: -1.0, y: 1.0)
        imageView?.transform = CGAffineTransform(scaleX: -1.0, y: 1.0)
    }
}

// MARK: - Dark mode support
extension SelectButton {
    override func traitCollectionDidChange(_ previousTraitCollection: UITraitCollection?) {
        setupBorderColor() // Updating borderColor for light/dark mode changes
    }
}
